package com.tencent.start.cgs.tools;

import java.io.DataOutput;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteOrder;

@SuppressWarnings("all")
public abstract class RandomAccessDataOutput extends OutputStream implements DataOutput {

    protected boolean bigEndian = true;
    protected boolean nativeByteOrder = (bigEndian == (ByteOrder.nativeOrder() == ByteOrder.BIG_ENDIAN));

    public final ByteOrder order() {
        return bigEndian ? ByteOrder.BIG_ENDIAN : ByteOrder.LITTLE_ENDIAN;
    }

    public final RandomAccessDataOutput order(ByteOrder bo) {
        bigEndian = (bo == ByteOrder.BIG_ENDIAN);
        nativeByteOrder = (bigEndian == (ByteOrder.nativeOrder() == ByteOrder.BIG_ENDIAN));
        return this;
    }

    @Override
    public void write(byte[] b) throws IOException {
        write(b, 0, b.length);
    }

    public final void writeBoolean(boolean v) throws IOException {
        write(v ? 1 : 0);
    }

    public final void writeByte(int v) throws IOException {
        write(v);
    }

    public final void writeChar(int v) throws IOException {
        if (nativeByteOrder) {
            write((v >>> 0) & 0xFF);
            write((v >>> 8) & 0xFF);
        } else {
            write((v >>> 8) & 0xFF);
            write((v >>> 0) & 0xFF);
        }
    }

    public final void writeShort(int v) throws IOException {
        if (nativeByteOrder) {
            write((v >>> 0) & 0xFF);
            write((v >>> 8) & 0xFF);
        } else {
            write((v >>> 8) & 0xFF);
            write((v >>> 0) & 0xFF);
        }
    }

    public final void writeInt(int v) throws IOException {
        if (nativeByteOrder) {
            write((v >>>  0) & 0xFF);
            write((v >>>  8) & 0xFF);
            write((v >>> 16) & 0xFF);
            write((v >>> 24) & 0xFF);
        } else {
            write((v >>> 24) & 0xFF);
            write((v >>> 16) & 0xFF);
            write((v >>>  8) & 0xFF);
            write((v >>>  0) & 0xFF);
        }
    }

    public final void writeLong(long v) throws IOException {
        if (nativeByteOrder) {
            write((int)(v >>>  0) & 0xFF);
            write((int)(v >>>  8) & 0xFF);
            write((int)(v >>> 16) & 0xFF);
            write((int)(v >>> 24) & 0xFF);
            write((int)(v >>> 32) & 0xFF);
            write((int)(v >>> 40) & 0xFF);
            write((int)(v >>> 48) & 0xFF);
            write((int)(v >>> 56) & 0xFF);
        } else {
            write((int)(v >>> 56) & 0xFF);
            write((int)(v >>> 48) & 0xFF);
            write((int)(v >>> 40) & 0xFF);
            write((int)(v >>> 32) & 0xFF);
            write((int)(v >>> 24) & 0xFF);
            write((int)(v >>> 16) & 0xFF);
            write((int)(v >>>  8) & 0xFF);
            write((int)(v >>>  0) & 0xFF);
        }
    }

    public final void writeFloat(float v) throws IOException {
        writeInt(Float.floatToIntBits(v));
    }

    public final void writeDouble(double v) throws IOException {
        writeLong(Double.doubleToLongBits(v));
    }

    public final void writeBytes(String s) throws IOException {
        write(s.getBytes());
    }

    public final void writeChars(String s) throws IOException {
        int clen = s.length();
        int blen = 2*clen;
        byte[] b = new byte[blen];
        char[] c = new char[clen];
        s.getChars(0, clen, c, 0);
        for (int i = 0, j = 0; i < clen; i++) {
            b[j++] = (byte)(c[i] >>> 8);
            b[j++] = (byte)(c[i] >>> 0);
        }
        write(b, 0, blen);
    }

    public final void writeUTF(String str) throws IOException {
        writeUTF(str, this);
    }

    public static void writeUTF(String str, DataOutput out) throws IOException {
        int strlen = str.length();
        int utflen = 0;
        int c, count = 0;

        /* use charAt instead of copying String to char array */
        for (int i = 0; i < strlen; i++) {
            c = str.charAt(i);
            if ((c >= 0x0001) && (c <= 0x007F)) {
                utflen++;
            } else if (c > 0x07FF) {
                utflen += 3;
            } else {
                utflen += 2;
            }
        }

        if (utflen > 65535)
            throw new IOException("encoded string too long: " + utflen + " bytes");

        byte[] bytearr = new byte[utflen+2];
        bytearr[count++] = (byte) ((utflen >>> 8) & 0xFF);
        bytearr[count++] = (byte) ((utflen >>> 0) & 0xFF);

        int i=0;
        for (i=0; i<strlen; i++) {
            c = str.charAt(i);
            if (!((c >= 0x0001) && (c <= 0x007F))) break;
            bytearr[count++] = (byte) c;
        }

        for (;i < strlen; i++){
            c = str.charAt(i);
            if ((c >= 0x0001) && (c <= 0x007F)) {
                bytearr[count++] = (byte) c;

            } else if (c > 0x07FF) {
                bytearr[count++] = (byte) (0xE0 | ((c >> 12) & 0x0F));
                bytearr[count++] = (byte) (0x80 | ((c >>  6) & 0x3F));
                bytearr[count++] = (byte) (0x80 | ((c >>  0) & 0x3F));
            } else {
                bytearr[count++] = (byte) (0xC0 | ((c >>  6) & 0x1F));
                bytearr[count++] = (byte) (0x80 | ((c >>  0) & 0x3F));
            }
        }
        out.write(bytearr, 0, utflen+2);
    }
}
